/** @file   ComboMeter.h
 * @brief   Declaration of ComboMeter class.
 * @version $Revision: 1.2 $
 * @date    $Date: 2006/07/13 16:26:21 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_COMBOMETER_H
#define H_WWW_COMBOMETER_H

#include "ObjectID.h"

// Forward declarations
struct BITMAP;

namespace WeWantWar {
  
// Forward declarations
class Statistics;


/** @class  ComboMeter
 * @brief   Keeps track of the combo-bonuses.
 * @author  Tomi Lamminsaari
 *
 * Each time the player kills an alien the advance() method must be called.
 * The update() method must be called once in every update round to calculate
 * the combo delays. 
 */
class ComboMeter
{
public:

  ///
  /// Constants, datatypes and static methods
  /// =======================================
  
  /** Currently achieved combo levels.
   */
  enum ComboLevel {
    /** No combo yet. */
    ENoCombo = 0,
    /** Triple kill */
    ETripleKill = 3,
    /** Butcher */
    EButcher = 5,
    /** Serial killed */
    ESerialKiller = 9,
    /** Alien massacre. */
    EAlienMassacre = 15
  };

  ///
  /// Constructors, destructor and operators
  /// ======================================

  /** Constructor.
   * @param   aStatistics       Reference to statistics object.
   */
  ComboMeter( Statistics& aStatistics );

  /** Destructor
   */
  ~ComboMeter();

private:
  ComboMeter( const ComboMeter& aAnother );
  ComboMeter& operator= ( const ComboMeter& aAnother );
public:

  ///
  /// New public methods
  /// ==================

  /** Increases the combo bonus level.
   * @param   aKilledObjectType Type of the killed object.
   */
  void advance( ObjectID::Type aKilledObjectType );
  
  /** Updates the combo bonus counters. Each time a combo bonuses have been
   * achieved, this method updates the statistics object passed to the
   * constructor.
   * @return  The combo that was granted to the player.
   */
  ComboLevel update();
  
  /** Draw this combo meter on given bitmap.
   * @param   aTargetBitmap     Pointer to target bitmap.
   */
  void redraw( BITMAP* aTargetBitmap );
  
  /** Returns the current combo level.
   * @return  Current combo level
   */
  ComboLevel currentComboLevel() const;

protected:
  ///
  /// Data members
  /// ============

  /** Combo timer. Calculates towards negative. */
  int iComboTimer;
  /** Current combo bonus level. */
  int iComboBonusSoFar;
  /** Reference to the statistics object. */
  Statistics& iStatistics;
};

};  // end of namespace

#endif
